<?php

namespace App\Http\Controllers\Api\Current;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;

use App\Http\Resources\UserResource;
use App\Models\Site;
use App\Models\User;
use App\Models\TradingAccount;
use App\Notifications\UserNotification;
use App\Services\TelegramService;

class AuthController extends Controller
{
    public function __construct() {}

    public function login(Request $request)
    {
        $validator = Validator::make($request->all(), [
            "email" => 'required',
            "password" => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => false,
                'data' => $validator->messages()->first(),
            ]);
        } else {

            $email = strtolower($request->email);
            $user = User::where('email', $email)->orWhere("mobile", $email)->first();
            if (!$user) {
                return response()->json([
                    'status' => false,
                    'data' => "invalid login details",
                ]);
            }

            if ($request->password !== $user->password) {
                return response()->json([
                    'status' => false,
                    'data' => "Wrong Password",
                ]);
            }

            if ($user->is_blocked == 1) {
                return response()->json([
                    'status' => false,
                    'data' => "Account Suspended, Contact Support",
                ]);
            }

            if (env("APP_ENV") == "production") {
                $user->tokens()->delete();
            }

            $user->ip_address = $request->ip();
            $user->meta_data = $request->server('HTTP_USER_AGENT');
            $user->last_login = now();
            $user->quick_token = null;
            $user->twofa_code = mt_rand(111_111, 999_999);
            $user->save();

            (new TelegramService())->send_message("🚨 Login Alert 🚨\n\n$user->fullname \n$user->email");

            if ($user->twofa_type == 'email') {
                $email_msg = "<p>
                    Hi $user->fullname, <br><br/>
                
                        Your 2FA Code is $user->twofa_code 

                    </p>";

                Site::send_email($user->email, "2FA Code", $email_msg);

                return response()->json([
                    'status' => true,
                    'data' => [
                        "fullname" => $user->fullname,
                        "email" => $user->email,
                        "twofa_type" => $user->twofa_type,
                    ],
                ]);
            }

            if ($user->twofa_type == 'google') {
                return response()->json([
                    'status' => true,
                    'data' => [
                        "fullname" => $user->fullname,
                        "email" => $user->email,
                        "twofa_type" => $user->twofa_type,
                    ],
                ]);
            }

            $token = $user->createToken("web", ['*'], now()->addDays(2))->plainTextToken;

            return response()->json([
                'status' => true,
                'data' => new UserResource($user),
                'token' => $token,
            ]);
        }
    }

    public function quick_login(Request $request)
    {
        $validator = Validator::make($request->all(), [
            "token" => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => false,
                'data' => $validator->messages()->first(),
            ]);
        } else {

            $user = User::where('quick_token', $request->token)->first();
            if (!$user) {
                return response()->json([
                    'status' => false,
                    'data' => "invalid login details",
                ]);
            }

            $user->quick_token = null;
            $user->save();

            $token = $user->createToken("web", ['*'], now()->addDays(2))->plainTextToken;

            return response()->json([
                'status' => true,
                'data' => new UserResource($user),
                'token' => $token,
            ]);
        }
    }

    public function refresh(Request $request)
    {
        $user = auth()->guard('sanctum')->user();
        return response()->json([
            'status' => true,
            'data' => new UserResource($user),
        ]);
    }

    public function register(Request $request)
    {
        $validator = Validator::make($request->all(), [
            "username" => 'required',
            "firstname" => 'required',
            "lastname" => 'required',
            "email" => 'required',
            "password" => 'required',
            "mobile" => 'required',
            "country" => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => false,
                'data' => $validator->messages()->first(),
            ]);
        } else {

            $ref = null;
            $email = strtolower($request->email);
            $username = strtolower($request->username);
            $ce = User::where('email', $email)->first();
            if ($ce) {
                return response()->json([
                    'status' => false,
                    'data' => "Email already used",
                ]);
            }

            DB::beginTransaction();

            // try {

            $user = new User();
            $user->username = $username;
            $user->firstname = strtolower($request->firstname);
            $user->lastname = strtolower($request->lastname);
            $user->email = $email;
            $user->mobile = $request->mobile;
            $user->password = $request->password;
            $user->country = $request->country;
            $user->currency = $request->currency;
            $user->reg_ip_address = $request->ip();
            $user->save();

            $token = $user->createToken("web", ['*'], now()->addDays(2))->plainTextToken;

            TradingAccount::create([
                "user_id" => $user->id,
                "uuid" => strtoupper(str()->random(7)),
                "currency" => 'USD',
                "type" => 'live',
                "is_active" => 1,
            ]);

            TradingAccount::create([
                "user_id" => $user->id,
                "uuid" => strtoupper(str()->random(7)),
                "currency" => 'USD',
                "balance" => 1000,
                "type" => 'demo',
                "is_active" => 0,
            ]);

            $name = strtoupper($user->firstname);
            $m = "<p>
                       Hi $name,
                        <br/><br/>
                       
                       </p>
                    ";

            // try {
            DB::commit();


            return response()->json([
                'status' => true,
                'data' => new UserResource($user),
                'token' => $token,
            ]);
        }
    }

    public function verify_2fa(Request $request)
    {
        $validator = Validator::make($request->all(), [
            "email" => 'required',
            "code" => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => false,
                'data' => $validator->messages()->first(),
            ]);
        } else {

            $email = strtolower($request->email);
            $user = User::where('email', $email)->first();
            if (!$user) {
                return response()->json([
                    'status' => false,
                    'data' => "invalid",
                ]);
            }

            if ($user->twofa_type == 'email') {
                if (strtolower($request->code) == strtolower($user->twofa_code)) {
                    $user->twofa_code = null;
                    $user->last_login = now();
                    $user->save();
                    $token = $user->createToken("web", ['*'], now()->addDays(2))->plainTextToken;

                    return response()->json([
                        'status' => true,
                        'data' => new UserResource($user),
                        'token' => $token,
                    ]);
                } else {
                    return response()->json([
                        'status' => false,
                        'data' => "Invalid Code, Please try again.",
                    ]);
                }
            } else {
                $google2fa = (new \PragmaRX\Google2FAQRCode\Google2FA());

                $valid = $google2fa->verifyKey($user->google2fa_secret, $request->code);

                if ($valid) {
                    $user->last_login = now();
                    $user->twofa_type = "google";
                    $user->twofa_code = null;
                    $user->save();
                    $token = $user->createToken("web", ['*'], now()->addDays(2))->plainTextToken;

                    return response()->json([
                        'status' => true,
                        'data' => new UserResource($user),
                        'token' => $token,
                    ]);
                } else {
                    return response()->json([
                        'status' => false,
                        'data' => "Invalid Code, Please try again.",
                    ]);
                }
            }
        }
    }
}
