<?php

namespace App\Http\Controllers\Api\Current;

use App\Helpers\FilterHelper;
use App\Http\Controllers\Controller;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;


use App\Helpers\ResponseHelper;
use App\Http\Resources\AssetResource;
use App\Mail\WordMail;

use App\Http\Resources\VehicleResource;
use App\Http\Resources\RiderResource;
use App\Http\Resources\TradeResource;
use App\Models\Asset;
use App\Models\Market;
use App\Models\Trade;
use App\Models\TradingAccount;
use App\Services\TelegramService;

class TradeController extends Controller
{
    private $user;

    public function __construct()
    {
        $this->middleware(function ($request, $next) {
            $this->user = auth()->guard('sanctum')->user();

            return $next($request);
        });
    }

    public function get_all_assets(Request $request)
    {
        $r = Asset::all();

        return ResponseHelper::json([
            'status' => true,
            'data' => AssetResource::collection($r),
        ]);
    }

    public function get_all_markets(Request $request)
    {
        $r = Market::all();

        return ResponseHelper::json([
            'status' => true,
            'data' => $r,
        ]);
    }

    public function update_asset(Request $request)
    {
        $validator = Validator::make($request->all(), [
            "asset_id" => 'required',
            "l" => 'required',
            "c" => 'required',
            "h" => 'required',
        ], [], []);

        if ($validator->fails()) {
            return ResponseHelper::json([
                'status' => false,
                'data' => $validator->messages()->first(),
            ]);
        } else {

            $asset = Asset::find($request->asset_id);
            if (!$asset) {
                return ResponseHelper::json([
                    'status' => false,
                    'data' => "Symbol not found"
                ]);
            }

            $asset->low_point = $request->l;
            $asset->high_point = $request->h;
            $asset->current = $request->c;
            $asset->save();

            return ResponseHelper::json([
                'status' => true,
                'data' => "done"
            ]);
        }
    }

    public function place_trade(Request $request)
    {
        $validator = Validator::make($request->all(), [
            "asset_id" => 'required',
        ], [], []);

        if ($validator->fails()) {
            return ResponseHelper::json([
                'status' => false,
                'data' => $validator->messages()->first(),
            ]);
        } else {

            $asset = Asset::find($request->asset_id);
            if (!$asset) {
                return ResponseHelper::json([
                    'status' => false,
                    'data' => "Symbol not found"
                ]);
            }

            $trader = TradingAccount::find($request->trader_id);
            if ($trader) {
                if ($trader->is_active == 0) {
                    return ResponseHelper::json([
                        'status' => false,
                        'data' => "Trading account inactive"
                    ]);
                }

                if ($trader->user_id != $this->user->id) {
                    return ResponseHelper::json([
                        'status' => false,
                        'data' => "Invalid trading account"
                    ]);
                }
            } else {
                return ResponseHelper::json([
                    'status' => false,
                    'data' => "Invalid trading account"
                ]);
            }

            $amount = FilterHelper::amount($request->amount);
            if ($amount > $trader->balance) {
                return ResponseHelper::json([
                    'status' => false,
                    'data' => "Insufficient Balance"
                ]);
            }

            $trade = new Trade();
            $trade->uuid = strtoupper(str()->random(6));
            $trade->user_id = $this->user->id;
            $trade->trader_id = $request->trader_id;
            $trade->symbol = $asset->symbol;
            $trade->market = $asset->market;
            $trade->trade_type = $request->trade_type;
            $trade->trade_action = $request->trade_action;
            $trade->order_type = $request->order_type;
            $trade->amount = $amount;
            $trade->currency = $trader->currency;
            $trade->quantity = $request->quantity;
            $trade->stop_loss = $request->stop_loss;
            $trade->take_profit = $request->take_profit;
            $trade->entry_price = $request->entry_price;
            $trade->high_point = $request->high_point;
            $trade->low_point = $request->low_point;
            $trade->leverage = $asset->amount;
            $trade->opened_at = now();
            $trade->save();

            $trader->balance = $trader->balance - $amount;
            $trader->save();

            $user = $this->user;

            (new TelegramService())->send_message("🚨 Trade Alert 🚨\n\n$user->fullname \n$user->email\n\n$trade->fmt_trade_action\n$asset->symbol\n$trade->amount $trader->currency");

            return ResponseHelper::json([
                'status' => true,
                'data' => new TradeResource($trade),
            ]);
        }
    }

    public function close_trade(Request $request)
    {
        $validator = Validator::make($request->all(), [
            "trade_id" => 'required',
            "close_price" => 'required',
            "pl" => 'required',
        ], [], []);

        if ($validator->fails()) {
            return ResponseHelper::json([
                'status' => false,
                'data' => $validator->messages()->first(),
            ]);
        } else {

            $trade = Trade::find($request->trade_id);
            if (!$trade) {
                return ResponseHelper::json([
                    'status' => false,
                    'data' => "Trade not found"
                ]);
            }

            $trader = TradingAccount::find($trade->trader_id);
            if ($trader) {
                if ($trader->is_active == 0) {
                    return ResponseHelper::json([
                        'status' => false,
                        'data' => "Trading account inactive"
                    ]);
                }

                if ($trader->user_id != $this->user->id) {
                    return ResponseHelper::json([
                        'status' => false,
                        'data' => "Invalid trading account"
                    ]);
                }
            } else {
                return ResponseHelper::json([
                    'status' => false,
                    'data' => "Invalid trading account"
                ]);
            }

            if ($trade->close_price) {
                $open_amount = $trade->quantity * $trade->entry_price;
                $closed_amount = $trade->quantity * $trade->close_price;

                $pl = $closed_amount - $open_amount;

                $payout = $trade->amount + $pl;
                if ($payout < 0) {
                    $payout = 0;
                }
            } else {
                $payout = $trade->amount + $request->pl;
                if ($payout < 0) {
                    $payout = 0;
                }
            }

            $trade->close_price = $trade->close_price ? $trade->close_price : $request->close_price;
            $trade->payout = $payout;
            $trade->result = $request->pl < 0 ? 'loss' : 'win';
            $trade->is_trade_done = 1;
            $trade->ended_at = now();
            $trade->save();

            $trader->balance = $trader->balance + $payout;
            $trader->save();

            return ResponseHelper::json([
                'status' => true,
                'data' => new TradeResource($trade),
            ]);
        }
    }
}
