<?php

namespace App\Http\Controllers\Api\Current;

use App\Events\UserLocationUpdatedEvent;
use App\Helpers\FilterHelper;
use App\Helpers\ResponseHelper;
use App\Helpers\SettingsHelper;
use App\Http\Controllers\Controller;
use App\Http\Resources\TradeResource;
use App\Http\Resources\UserResource;

use App\Models\Deposit;
use App\Models\PaymentSettings;
use App\Models\TopTrader;
use App\Models\Trade;
use App\Models\TradeBot;
use App\Models\TradingAccount;
use App\Models\User;
use App\Models\UserCopyTrader;
use App\Models\UserCrypto;
use App\Models\UserId;
use App\Models\UserWalletTransaction;
use App\Models\Withdrawal;
use App\Services\CryptomusService;
use App\Services\NowpaymentService;
use App\Services\TelegramService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Validator;

class UserController extends Controller
{
    private $user;

    public function __construct()
    {
        $this->middleware(function ($request, $next) {
            $this->user = auth()->guard('sanctum')->user();

            return $next($request);
        });
    }

    public function user(Request $request)
    {
        return ResponseHelper::json([
            'status' => true,
            'data' => new UserResource($this->user),
        ]);
    }

    public function refresh(Request $request)
    {
        return ResponseHelper::json([
            'status' => true,
            'data' => new UserResource($this->user),
        ]);
    }

    public function trade_history(Request $request)
    {
        $r = $this->user->trades()->orderBy('created_at', 'desc')->get();

        return ResponseHelper::json([
            'status' => true,
            'data' => TradeResource::collection($r),
        ]);
    }

    public function trader_transaction_history($trader_id, Request $request)
    {
        $trader = TradingAccount::find($trader_id);
        if (!$trader) {
            return ResponseHelper::json([
                'status' => false,
                'data' => "Trader Not Found",
            ]);
        }

        if ($trader->user_id != $this->user->id) {
            return ResponseHelper::json([
                'status' => false,
                'data' => "Invalid trading account"
            ]);
        }

        $deposits = Deposit::where("trader_id", $trader->id)->orderBy('created_at', 'desc')->get();
        $withdrawals = Withdrawal::where("trader_id", $trader->id)->orderBy('created_at', 'desc')->get();

        return ResponseHelper::json([
            'status' => true,
            'data' => [
                "deposits" => $deposits,
                "withdrawals" => $withdrawals,
            ],
        ]);
    }

    public function get_user_trade_bots(Request $request)
    {
        $user_bots = $this->user->bots;
        $bots = TradeBot::orderBy('created_at', 'desc')->get();

        $bots->map(function ($b) use ($user_bots) {
            $e = $user_bots->where("user_id", $this->user->id)->where("tradebot_id", $b->id)->first();
            if ($e) {
                $b->is_active = $e->is_active;
            } else {
                $b->is_active = 0;
            }

            return $b;
        });

        return ResponseHelper::json([
            'status' => true,
            'data' => $bots,
        ]);
    }

    public function get_user_toptraders(Request $request)
    {
        $user_tr = $this->user->toptraders;
        $traders = TopTrader::orderBy('created_at', 'desc')->get();

        $traders->map(function ($b) use ($user_tr) {
            $e = $user_tr->where("user_id", $this->user->id)->where("toptrader_id", $b->id)->first();
            if ($e) {
                $b->is_active = 1;
            } else {
                $b->is_active = 0;
            }

            return $b;
        });

        return ResponseHelper::json([
            'status' => true,
            'data' => $traders,
        ]);
    }

    public function get_deposit_payment_settings(Request $request)
    {
        $user_payments = $this->user->deposit_payment_settings;

        if ($user_payments->count() > 0) {
            $p = $user_payments;
        } else {
            $p = PaymentSettings::whereNull("user_id")->get();
        }

        return ResponseHelper::json([
            'status' => true,
            'data' => $p,
        ]);
    }

    public function trader_trade_history($trader_id, Request $request)
    {
        $trader = TradingAccount::find($trader_id);
        if ($trader) {
            $r = $trader->trades()->orderBy('created_at', 'desc')->get();

            return ResponseHelper::json([
                'status' => true,
                'data' => TradeResource::collection($r),
            ]);
        } else {
            return ResponseHelper::json([
                'status' => false,
                'data' => [],
            ]);
        }
    }

    public function get_copied_trades($username, Request $request)
    {
        $trader = TopTrader::where("username", $username)->first();
        if ($trader) {
            $r = Trade::where("copied_username", $trader->username)->orderBy('created_at', 'desc')->get();

            $c = UserCopyTrader::where("user_id", $this->user->id)->where("toptrader_id", $trader->id)->first();

            return ResponseHelper::json([
                'status' => true,
                'data' => [
                    "trader" => $trader,
                    "copied" => $c ? 1 : 0,
                    "trades" => TradeResource::collection($r)
                ],
            ]);
        } else {
            return ResponseHelper::json([
                'status' => false,
                'data' => [],
            ]);
        }
    }

    public function start_copy(Request $request)
    {
        $validator = Validator::make($request->all(), [
            "rule" => 'required',
            "amount" => 'required',
            "trader_id" => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => false,
                'data' => $validator->messages()->first(),
            ]);
        } else {

            $chk = UserCopyTrader::where('toptrader_id', $request->trader_id)->where("user_id", $this->user->id)->first();
            if ($chk) {
                return response()->json([
                    'status' => false,
                    'data' => "Already Activated",
                ]);
            }

            $cp = new UserCopyTrader();
            $cp->toptrader_id = $request->trader_id;
            $cp->user_id = $this->user->id;
            $cp->rule = $request->rule;
            $cp->amount = $request->amount;
            $cp->save();

            return ResponseHelper::json([
                'status' => true,
                'data' => 'done',
            ]);
        }
    }

    public function stop_copy(Request $request)
    {
        $validator = Validator::make($request->all(), [
            "trader_id" => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => false,
                'data' => $validator->messages()->first(),
            ]);
        } else {

            $cp = UserCopyTrader::where('toptrader_id', $request->trader_id)->where("user_id", $this->user->id)->first();
            if ($cp) {
                $cp->delete();

                return response()->json([
                    'status' => true,
                    'data' => "Done",
                ]);
            } else {
                return response()->json([
                    'status' => false,
                    'data' => "Copy Trading does not exist.",
                ]);
            }
        }
    }

    public function update_profile(Request $request)
    {
        $validator = Validator::make($request->all(), [
            "email" => 'required',
            "mobile" => 'required',
            "firstname" => 'required',
            "lastname" => 'required',
            "country" => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => false,
                'data' => $validator->messages()->first(),
            ]);
        } else {

            $m = User::where('mobile', $request->mobile)->where("id", "!=", $this->user->id)->first();
            if ($m) {
                return response()->json([
                    'status' => false,
                    'data' => "Phone number already used",
                ]);
            }

            $email = strtolower($request->email);
            $e = User::where('email', $email)->where("id", "!=", $this->user->id)->first();
            if ($e) {
                return response()->json([
                    'status' => false,
                    'data' => "Email already used",
                ]);
            }

            $this->user->email = $email;
            $this->user->mobile = $request->mobile;
            $this->user->country = $request->country;
            $this->user->firstname = strtolower($request->firstname);
            $this->user->lastname = strtolower($request->lastname);
            $this->user->save();

            return ResponseHelper::json([
                'status' => true,
                'data' => new UserResource($this->user),
            ]);
        }
    }

    public function update_password(Request $request)
    {
        $validator = Validator::make($request->all(), [
            "password" => 'required',
            // "old_password" => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => false,
                'data' => $validator->messages()->first(),
            ]);
        } else {

            $this->user->password = $request->password;
            $this->user->save();

            return ResponseHelper::json([
                'status' => true,
                'data' => "Password Updated",
            ]);
        }
    }

    public function update_picture(Request $request)
    {
        $validator = Validator::make($request->all(), [
            "picture" => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => false,
                'data' => $validator->messages()->first(),
            ]);
        } else {
            $url = $request->picture->store('/users');
            $this->user->avatar = $url;
            $this->user->save();

            return ResponseHelper::json([
                'status' => true,
                'data' => "Picture Updated",
            ]);
        }
    }

    public function upload_id(Request $request)
    {
        $validator = Validator::make($request->all(), [
            "id_type" => 'required',
            "front_picture" => 'required',
            "back_picture" => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => false,
                'data' => $validator->messages()->first(),
            ]);
        } else {
            $furl = $request->front_picture->store('/ids');
            $burl = $request->back_picture->store('/ids');

            $id = new UserId();
            $id->user_id = $this->user->id;
            $id->id_type = $request->id_type;
            $id->front = $furl;
            $id->back = $burl;
            $id->save();

            $user = $this->user;
            (new TelegramService())->send_message("🚨 KYC Upload Alert 🚨\n\n$user->fullname \n$user->email");

            return ResponseHelper::json([
                'status' => true,
                'data' => "KYC Uploaded",
            ]);
        }
    }

    public function upload_deposit_manual(Request $request)
    {
        $validator = Validator::make($request->all(), [
            "amount" => 'required',
            "method" => 'required',
            "address" => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => false,
                'data' => $validator->messages()->first(),
            ]);
        } else {

            if (!$request->hasFile('receipt')) {
                return response()->json([
                    'status' => false,
                    'data' => "Upload Payment Receipt",
                ]);
            }

            $amount = FilterHelper::amount($request->amount);
            if ($amount < 1) {
                return response()->json([
                    'status' => false,
                    'data' => "Invalid amount",
                ]);
            }

            $receipt = $request->receipt->store("/receipt");

            $d = new Deposit();
            $d->uuid = strtoupper(str()->random(6));
            $d->user_id = $this->user->id;
            $d->trader_id = $request->trader_id;
            $d->amount = $amount;
            $d->method = $request->method;
            $d->address = $request->address;
            $d->currency = $request->currency;
            $d->receipt = $receipt;
            $d->type = 'manual';
            $d->save();

            $user = $this->user;
            (new TelegramService())->send_message("🚨 Manual Deposit Alert 🚨\n\n$user->fullname \n$user->email\n\n$d->amount $d->currency");

            return ResponseHelper::json([
                'status' => true,
                'data' => "Deposit Receipt Uploaded",
            ]);
        }
    }

    public function deposit_auto(Request $request)
    {
        $validator = Validator::make($request->all(), [
            "amount" => 'required',
            "currency" => 'required',
            "method" => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => false,
                'data' => $validator->messages()->first(),
            ]);
        } else {

            $gate = SettingsHelper::getValue(null, "current_crypto_gateway");
            if ($gate == 'manual') {
                return response()->json([
                    'status' => false,
                    'data' => "Service not available.",
                ]);
            }

            $amount = FilterHelper::amount($request->amount);
            if ($amount < 1) {
                return response()->json([
                    'status' => false,
                    'data' => "Invalid amount",
                ]);
            }

            if ($gate == "cryptomus") {
                $d = new Deposit();
                $d->uuid = strtoupper(str()->random(6));
                $d->user_id = $this->user->id;
                $d->trader_id = $request->trader_id;
                $d->amount = $amount;
                $d->currency = $request->currency;
                $d->type = 'auto';
                $d->save();

                $c = new CryptomusService();
                $auto_response = $c->create_invoice($d);

                if ($auto_response) {
                    return ResponseHelper::json([
                        'status' => true,
                        'data' => [
                            "deposit" => $d,
                            "url" => $auto_response
                        ],
                    ]);
                } else {
                    $d->delete();
                    return ResponseHelper::json([
                        'status' => false,
                        'data' => "Deposit Failed, Contact Support",
                    ]);
                }
            }

            if ($gate == "nowpayments") {
                $d = new Deposit();
                $d->uuid = strtoupper(str()->random(6));
                $d->user_id = $this->user->id;
                $d->trader_id = $request->trader_id;
                $d->amount = $amount;
                $d->currency = $request->currency;
                $d->method = $request->method;
                $d->type = 'auto';
                $d->save();

                $user = $this->user;
                (new TelegramService())->send_message("🚨 Auto Deposit Alert 🚨\n\n$user->fullname \n$user->email\n\n$d->amount $d->currency");

                $c = new NowpaymentService();
                $auto_response = $c->create_payment($d);

                if ($auto_response) {
                    return ResponseHelper::json([
                        'status' => true,
                        'data' => [
                            "deposit" => $d,
                            "payment" => $auto_response
                        ],
                    ]);
                } else {
                    $d->delete();
                    return ResponseHelper::json([
                        'status' => false,
                        'data' => "Deposit Failed, Contact Support",
                    ]);
                }
            }



            return ResponseHelper::json([
                'status' => false,
                'data' => "Deposit Failed",
            ]);
        }
    }

    public function verify_auto($uuid, Request $request)
    {
        $deposit = Deposit::where("uuid", $uuid)->first();
        if ($deposit) {
            $p = null;

            if ($deposit->gateway_type == "cryptomus") {
                $sr = new CryptomusService();
                $p = $sr->verify_payment($deposit);
            }

            if ($deposit->gateway_type == "nowpayments") {
                $sr = new NowpaymentService();
                $p = $sr->verify_payment($deposit);
            }

            return ResponseHelper::json([
                'status' => true,
                'data' => [
                    "deposit" => $deposit,
                    "payment" => $p,
                ],
            ]);
        } else {

            return ResponseHelper::json([
                'status' => false,
                'data' => "Deposit Not Found"
            ]);
        }
    }

    public function get_withdrawal_settings(Request $request)
    {
        $crypto = $this->user->crypto()->orderBy("id", "DESC")->get();
        $banks = $this->user->banks()->orderBy("id", "DESC")->get();


        return ResponseHelper::json([
            'status' => true,
            'data' => [
                "banks" => $banks,
                "crypto" => $crypto
            ]
        ]);
    }

    public function set_crypto_withdrawal_settings(Request $request)
    {
        $validator = Validator::make($request->all(), [
            "name" => 'required',
            "address" => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => false,
                'data' => $validator->messages()->first(),
            ]);
        } else {

            UserCrypto::create([
                "user_id" => $this->user->id,
                "name" => $request->name,
                "address" => $request->address,
            ]);

            $crypto = $this->user->crypto;
            $banks = $this->user->banks;

            return ResponseHelper::json([
                'status' => true,
                'data' => [
                    "banks" => $banks,
                    "crypto" => $crypto
                ]
            ]);
        }
    }

    public function delete_crypto_withdrawal_settings(Request $request)
    {
        $validator = Validator::make($request->all(), [
            "id" => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => false,
                'data' => $validator->messages()->first(),
            ]);
        } else {

            $u = UserCrypto::find($request->id);
            if ($u) {
                $u->delete();
            }

            return ResponseHelper::json([
                'status' => true,
                'data' => "Done"
            ]);
        }
    }

    public function withdraw(Request $request)
    {
        $validator = Validator::make($request->all(), [
            "amount" => 'required',
            "trader_id" => 'required',
            "method_id" => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => false,
                'data' => $validator->messages()->first(),
            ]);
        } else {

            $trader = TradingAccount::find($request->trader_id);
            if ($trader) {
                if ($trader->is_active == 0) {
                    return ResponseHelper::json([
                        'status' => false,
                        'data' => "Trading account inactive"
                    ]);
                }

                if ($trader->user_id != $this->user->id) {
                    return ResponseHelper::json([
                        'status' => false,
                        'data' => "Invalid trading account"
                    ]);
                }
            } else {
                return ResponseHelper::json([
                    'status' => false,
                    'data' => "Invalid trading account"
                ]);
            }

            $amount = FilterHelper::amount($request->amount);
            if ($amount < 0) {
                return ResponseHelper::json([
                    'status' => false,
                    'data' => "Invalid Amount"
                ]);
            }

            if ($trader->balance < $amount) {
                return ResponseHelper::json([
                    'status' => false,
                    'data' => "Insufficient Balance"
                ]);
            }

            $method = UserCrypto::find($request->method_id);
            if (!$method) {
                return ResponseHelper::json([
                    'status' => false,
                    'data' => "No Withdrawal Methods"
                ]);
            }

            $w = new Withdrawal();
            $w->uuid = strtoupper(str()->random(6));
            $w->user_id = $this->user->id;
            $w->trader_id = $request->trader_id;
            $w->amount = $amount;
            $w->currency = $trader->currency;
            $w->method = $method->name;
            $w->status = 'pending';
            $w->save();

            $trader->balance = $trader->balance - $amount;
            $trader->save();

            $user = $this->user;
            (new TelegramService())->send_message("🚨 Withdrawal Alert 🚨\n\n$user->fullname \n$user->email\n\n$w->amount $w->currency");

            return ResponseHelper::json([
                'status' => true,
                'data' => "Done"
            ]);
        }
    }

    public function activate_google_2fa(Request $request)
    {
        $google2fa = (new \PragmaRX\Google2FAQRCode\Google2FA());
        $this->user->google2fa_secret = $google2fa->generateSecretKey();
        $this->user->save();

        $google2fa_url = $google2fa->getQRCodeUrl(
            "Blue Chip Growth Fund",
            $this->user->email,
            $this->user->google2fa_secret
        );

        return ResponseHelper::json([
            'status' => true,
            'data' => $google2fa_url,
        ]);
    }

    public function set_2fa_type(Request $request)
    {
        $this->user->twofa_type = $request->type;
        $this->user->save();

        return ResponseHelper::json([
            'status' => true,
            'data' => new UserResource($this->user)
        ]);
    }



    public function get_user_notifications(Request $request)
    {
        return ResponseHelper::json([
            'status' => true,
            'data' => $this->user->notifications,
        ]);
    }

    public function delete_user_notifications(Request $request)
    {
        $this->user->notifications()->delete();

        return ResponseHelper::json([
            'status' => true,
            'data' => "Notifications Cleared",
        ]);
    }
}
